/* $Id: jpggen.cpp $ */
/** @file
 * JPEG image generator.
 *
 * Generates a solid component color JPEG image. Width and height must be a multiple of 8.
 */

/*
 * Copyright (C) 2013-2025 Oracle and/or its affiliates.
 *
 * This file is part of VirtualBox base platform packages, as
 * available from https://www.virtualbox.org.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, in version 3 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses>.
 *
 * SPDX-License-Identifier: GPL-3.0-only
 */

#ifndef JPGGENTEST

#include <string.h>

#include <iprt/alloc.h>
#include <iprt/asm.h>
#include <iprt/cdefs.h>
#include <iprt/errcore.h>
#include <iprt/types.h>

#endif /* !JPGGENTEST */

#pragma pack(1)
typedef struct SOF0Header
{
    uint16_t u16Marker;
    uint16_t u16LengthBE;
    uint8_t  u8Precision;
    uint16_t u16HeightBE;
    uint16_t u16WidthBE;
    uint16_t u8NumComponents;
} SOF0Header;
#pragma pack()

/*
 * JPEG image parts.
 */
static const uint8_t au8SOI[] =
{
    0xff, 0xd8
};

static const uint8_t au8DQT_Id0[] =
{
    0xff, 0xdb, 0x00, 0x43, 0x00, 0x08, 0x06, 0x06, 0x07, 0x06, 0x05, 0x08, 0x07, 0x07, 0x07, 0x09,
    0x09, 0x08, 0x0a, 0x0c, 0x14, 0x0d, 0x0c, 0x0b, 0x0b, 0x0c, 0x19, 0x12, 0x13, 0x0f, 0x14, 0x1d,
    0x1a, 0x1f, 0x1e, 0x1d, 0x1a, 0x1c, 0x1c, 0x20, 0x24, 0x2e, 0x27, 0x20, 0x22, 0x2c, 0x23, 0x1c,
    0x1c, 0x28, 0x37, 0x29, 0x2c, 0x30, 0x31, 0x34, 0x34, 0x34, 0x1f, 0x27, 0x39, 0x3d, 0x38, 0x32,
    0x3c, 0x2e, 0x33, 0x34, 0x32
};

static const uint8_t au8DQT_Id1[] =
{
    0xff, 0xdb, 0x00, 0x43, 0x01, 0x09, 0x09, 0x09, 0x0c, 0x0b, 0x0c, 0x18, 0x0d, 0x0d, 0x18, 0x32,
    0x21, 0x1c, 0x21, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32,
    0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32,
    0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32, 0x32,
    0x32, 0x32, 0x32, 0x32, 0x32
};

/* Important to use h_samp = 2, v_samp = 1 for the first component, i.e. the same as real webcams. */
static const uint8_t au8SOF0[] =
{
    0xff, 0xc0, 0x00, 0x11, 0x08, 0x01, 0xe0, 0x02, 0x80, 0x03, 0x01, 0x21, 0x00, 0x02, 0x11, 0x01,
    0x03, 0x11, 0x01
};

static const uint8_t au8DHT_DC_Id0[] =
{
    0xff, 0xc4, 0x00, 0x1f, 0x00, 0x00, 0x01, 0x05, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
    0x0b
};

static const uint8_t au8DHT_AC_Id0[] =
{
    0xff, 0xc4, 0x00, 0xb5, 0x10, 0x00, 0x02, 0x01, 0x03, 0x03, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04,
    0x04, 0x00, 0x00, 0x01, 0x7d, 0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12, 0x21, 0x31, 0x41,
    0x06, 0x13, 0x51, 0x61, 0x07, 0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08, 0x23, 0x42, 0xb1,
    0xc1, 0x15, 0x52, 0xd1, 0xf0, 0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0a, 0x16, 0x17, 0x18, 0x19,
    0x1a, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43, 0x44,
    0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x63, 0x64,
    0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x83, 0x84,
    0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0xa2,
    0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9,
    0xba, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
    0xd8, 0xd9, 0xda, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xf1, 0xf2, 0xf3,
    0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa
};

static const uint8_t au8DHT_DC_Id1[] =
{
    0xff, 0xc4, 0x00, 0x1f, 0x01, 0x00, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
    0x0b
};

static const uint8_t au8DHT_AC_Id1[] =
{
    0xff, 0xc4, 0x00, 0xb5, 0x11, 0x00, 0x02, 0x01, 0x02, 0x04, 0x04, 0x03, 0x04, 0x07, 0x05, 0x04,
    0x04, 0x00, 0x01, 0x02, 0x77, 0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21, 0x31, 0x06, 0x12,
    0x41, 0x51, 0x07, 0x61, 0x71, 0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42, 0x91, 0xa1, 0xb1, 0xc1,
    0x09, 0x23, 0x33, 0x52, 0xf0, 0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34, 0xe1, 0x25, 0xf1,
    0x17, 0x18, 0x19, 0x1a, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43,
    0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x63,
    0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x82,
    0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
    0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5,
    0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xf2, 0xf3,
    0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa
};

static const uint8_t au8SOS[] =
{
    0xff, 0xda, 0x00, 0x0c, 0x03, 0x01, 0x00, 0x02, 0x11, 0x03, 0x11, 0x00, 0x3f, 0x00
};

/*
 * Image consists of au8ImageBegin, repeated au8ImageBody and au8ImageEnd.
 * Total number of bytes is 4 + 5 * number of 8x8 tiles.
 *
 * The au8ImageBody bytes must be repeated:
 * cBodyRepeat = tiles - 2
 *
 * If the image width or height in tiles is not even, then it
 * must be rounded up to the next even value: i.e. 15->16.
 */

#define JPG_GEN_IMAGE_BEGIN_SIZE (4 + 5)

static const uint8_t au8ImageBeginRed[JPG_GEN_IMAGE_BEGIN_SIZE] =
{
    0xE2, 0xE8, 0xAF, 0x99, 0x3F, 0x71, 0x0A, 0x28, 0x00
};

static const uint8_t au8ImageBeginGreen[JPG_GEN_IMAGE_BEGIN_SIZE] =
{
    0xD6, 0xA2, 0xBF, 0x34, 0x3F, 0x20, 0x0A, 0x28, 0x00
};

static const uint8_t au8ImageBeginBlue[JPG_GEN_IMAGE_BEGIN_SIZE] =
{
    0xF1, 0xCA, 0x2B, 0xF7, 0x13, 0xCC, 0x0A, 0x28, 0x00
};

static const uint8_t au8ImageBody[] =
{
    0xA2, 0x80, 0x0A, 0x28, 0x00
};

static const uint8_t au8ImageEnd[] =
{
    0xA2, 0x80, 0x0A, 0x28, 0x03
};

static const uint8_t au8EOI[] =
{
    0xff, 0xd9
};

/*
 * Returned memory will be freed with RTMemFree.
 */
int JPGGenCreate(uint16_t u16Width, uint16_t u16Height, int iComponent, uint8_t **ppu8Data, uint32_t *pcbData)
{
    if (   u16Width == 0 || (u16Width & 0x7) != 0
        || u16Height == 0 ||(u16Height & 0x7) != 0)
    {
        return VERR_INVALID_PARAMETER;
    }

    uint32_t cTilesX = (uint32_t)u16Width >> 3;
    uint32_t cTilesY = (uint32_t)u16Height >> 3;

    cTilesX = (cTilesX + 1) & ~1;
    cTilesY = (cTilesY + 1) & ~1;

    uint32_t cTiles = cTilesX * cTilesY;

    uint32_t cImageBodyRepeats = cTiles - 2;

    uint32_t cbData =  sizeof(au8SOI) +
                       sizeof(au8DQT_Id0) +
                       sizeof(au8DQT_Id1) +
                       sizeof(au8SOF0) +
                       sizeof(au8DHT_DC_Id0) +
                       sizeof(au8DHT_AC_Id0) +
                       sizeof(au8DHT_DC_Id1) +
                       sizeof(au8DHT_AC_Id1) +
                       sizeof(au8SOS) +
                       JPG_GEN_IMAGE_BEGIN_SIZE +
                       sizeof(au8ImageBody) * cImageBodyRepeats +
                       sizeof(au8ImageEnd) +
                       sizeof(au8EOI);

    uint8_t *pu8Data = (uint8_t *)RTMemAlloc(cbData);
    if (pu8Data == NULL)
    {
        return VERR_NO_MEMORY;
    }

    uint8_t *pu8Dst = pu8Data;

    /* Start of image */
    memcpy(pu8Dst, au8SOI, sizeof(au8SOI));
    pu8Dst += sizeof(au8SOI);

    /* Quantization tables. */
    memcpy(pu8Dst, au8DQT_Id0, sizeof(au8DQT_Id0));
    pu8Dst += sizeof(au8DQT_Id0);
    memcpy(pu8Dst, au8DQT_Id1, sizeof(au8DQT_Id1));
    pu8Dst += sizeof(au8DQT_Id1);

    /* Start of frame. */
    SOF0Header *pSOF0Header = (SOF0Header *)pu8Dst;

    memcpy(pu8Dst, au8SOF0, sizeof(au8SOF0));
    pu8Dst += sizeof(au8SOF0);

    pSOF0Header->u16HeightBE = RT_H2BE_U16(u16Height);
    pSOF0Header->u16WidthBE = RT_H2BE_U16(u16Width);

    /* Huffman tables */
    memcpy(pu8Dst, au8DHT_DC_Id0, sizeof(au8DHT_DC_Id0));
    pu8Dst += sizeof(au8DHT_DC_Id0);
    memcpy(pu8Dst, au8DHT_AC_Id0, sizeof(au8DHT_AC_Id0));
    pu8Dst += sizeof(au8DHT_AC_Id0);
    memcpy(pu8Dst, au8DHT_DC_Id1, sizeof(au8DHT_DC_Id1));
    pu8Dst += sizeof(au8DHT_DC_Id1);
    memcpy(pu8Dst, au8DHT_AC_Id1, sizeof(au8DHT_AC_Id1));
    pu8Dst += sizeof(au8DHT_AC_Id1);

    /* Start of scan. */
    memcpy(pu8Dst, au8SOS, sizeof(au8SOS));
    pu8Dst += sizeof(au8SOS);

    /* Image */
    const uint8_t *pu8 = iComponent == 2? &au8ImageBeginRed[0]:
                         iComponent == 1? &au8ImageBeginGreen[0]:
                                          &au8ImageBeginBlue[0];
    memcpy(pu8Dst, pu8, JPG_GEN_IMAGE_BEGIN_SIZE);
    pu8Dst += JPG_GEN_IMAGE_BEGIN_SIZE;

    while (cImageBodyRepeats--)
    {
        memcpy(pu8Dst, au8ImageBody, sizeof(au8ImageBody));
        pu8Dst += sizeof(au8ImageBody);
    }

    memcpy(pu8Dst, au8ImageEnd, sizeof(au8ImageEnd));
    pu8Dst += sizeof(au8ImageEnd);

    /* End of image. */
    memcpy(pu8Dst, au8EOI, sizeof(au8EOI));
    pu8Dst += sizeof(au8EOI);

    *ppu8Data = pu8Data;
    *pcbData = (uint32_t)((uintptr_t)pu8Dst - (uintptr_t)pu8Data);

    Assert(*pcbData == cbData);

    return VINF_SUCCESS;
}
